unit Analysis;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, ComCtrls, Main;

type
  TAnalysisForm = class(TForm)
    ControlsBox: TGroupBox;
    Panel1: TPanel;
    ModelBox: TGroupBox;
    InpDelayEdit: TEdit;
    OutGainEdit: TEdit;
    DampEdit: TEdit;
    RefEdit: TEdit;
    OutGainUpDown: TUpDown;
    DampUpDown: TUpDown;
    RefUpDown: TUpDown;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    Label7: TLabel;
    AutoFitBtn: TButton;
    Label8: TLabel;
    CloseBtn: TButton;
    InDelayUpDown: TUpDown;
    OpenDialog1: TOpenDialog;
    OpenBtn: TButton;
    FileEdit: TEdit;
    SaveDialog1: TSaveDialog;
    SaveParamBtn: TButton;
    ParamFileNameEdit: TEdit;
    RetrieveParmsBtn: TButton;
    TrackDataPanel: TPanel;
    ModelFitPanel: TPanel;
    ModelPaint: TPaintBox;
    RunPaintBox: TPaintBox;
    ErrFitEdit: TEdit;
    Label9: TLabel;
    ErrPredEdit: TEdit;
    Label10: TLabel;
    AnalysisKeyBox: TGroupBox;
    Shape1: TShape;
    Shape2: TShape;
    Shape3: TShape;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Shape4: TShape;
    Shape5: TShape;
    Shape6: TShape;
    Label11: TLabel;
    Label12: TLabel;
    Label13: TLabel;
    DiffEdit: TEdit;
    Label14: TLabel;
    DistNoEdit: TEdit;
    Label15: TLabel;
    Rescale1: TButton;
    procedure CloseBtnClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure InDelayUpDownClick(Sender: TObject; Button: TUDBtnType);
    procedure OutGainUpDownClick(Sender: TObject; Button: TUDBtnType);
    procedure DampUpDownClick(Sender: TObject; Button: TUDBtnType);
    procedure RefUpDownClick(Sender: TObject; Button: TUDBtnType);
    procedure AutoFitBtnClick(Sender: TObject);
    procedure OpenBtnClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure SaveParamBtnClick(Sender: TObject);
    procedure RetrieveParmsBtnClick(Sender: TObject);
    procedure Rescale1Click(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    procedure InitAnModel;
    procedure CalcError;
    procedure RunModel;
    procedure FitParameter(var Param: Real; pMin,pMax: Integer);
    procedure FitAll;
    procedure SaveParameters;
    procedure SaveMultiRunParameters;
    procedure SaveTheParameters;
    procedure ReadParameters;
    procedure PlotRunGraph(Color: Integer;
      var Data: DataArray; YRange: Real; Bitmap: TBitmap);
    procedure PlotRun;
    procedure PlotModel(PlotRange: Real);
    procedure ReadData;
  end;

var
  AnalysisForm: TAnalysisForm;
  ModelBitmap: TBitmap;
  GainSet, DelaySet, DampSet, RefSet, pp: Real;
  LastGain, LastDelay, LastRef, LastDamp: Real;
  ErrFit, ErrPred, dt, OldRMS, TargetPos, CursorPos: Real;
  Damping, Gain, mPerc, mDelP, mHand, mHandVel, mCurs, mErr: Real;
  DelayBuffer: array[0..255] of Real;
  ErrorList: array[1..6] of Real;
  InPtr, OutPtr, RepRate, TimeLag, ModelRef: Integer;
  NumRuns, RunNumber, DiffLevel, DistNo, TargetRange: Integer;
  ModelMouseVal, ModelPercep, ModelDelPerc, ModelCursor, ModelHandle,
    FitErr, PredictErr, ErrorTimesFive: DataArray;
  SaveDamp, SaveGain, SaveLag, SaveRef: String;
  RootName, PathName, FullName, FullPath, FullParPath, TxtFileName,
    ParFilename: String;
  MyFile: Text;
  TimesFive: Boolean;

implementation

{$R *.dfm}

var
  RunBitmap: TBitmap;

procedure TAnalysisForm.InitAnModel;
begin
  GainSet := OutGainUpDown.Position;
  DelaySet := InDelayUpDown.Position;
  Dampset := DampUpDown.Position;
  RefSet := RefUpDown.Position;
  ModelMouseVal[1] := 0;

end;

procedure TAnalysisForm.CalcError;
var
  I: Integer;
begin
  ErrFit := 0.0;
  ErrPred := 0.0;
  for I := 1 to LastData do
    begin
      ErrFit := ErrFit + Sqr(FitErr[I]);
      ErrPred := ErrPred + Sqr(PredictErr[I]);
    end;
  ErrFit := 100 * Sqrt(ErrFit/LastData)/1000.0;
  ErrPred := 100 * Sqrt(ErrPred/LastData)/1000.0;
  ErrFitEdit.Text := Format('%6.3f',[ErrFit]);
  ErrPredEdit.Text := Format('%6.3f',[ErrPred]);
end;

function TransportLag(p: double; timelag: integer): double;
begin
  InPtr := (InPtr + 1) and 255;
  DelayBuffer[InPtr] := p;
  OutPtr := (InPtr - TimeLag + 256) and 255;
  Result := DelayBuffer[OutPtr];
end;

procedure TAnalysisForm.RunModel;
var
  T: Integer;
begin
  FillChar(DelayBuffer,SizeOf(DelayBuffer),0);
  Damping := 0.001*DampSet;
  Gain := 0.1*GainSet;
  TimeLag := Round(DelaySet);
  ModelRef := Round(RefSet/10);
  mHand := MouseVal[1];
  mCurs := mHand;
  T := 1;
  while T <= LastData do
  begin
    mPerc := (mCurs - TargetVal[T]);
    mDelP := TransportLag(mPerc,TimeLag);
    mErr := ModelRef - mDelP;
    mHand := mHand +(Gain*mErr - Damping*mHand)*dt;
    if mHand > 1000 then mHand := 1000
    else if mHand < -1000 then mHand := -1000;
    mCurs := mHand;
    ModelPercep[T] := mPerc;
    ModelDelPerc[T] := mDelP;
    ModelCursor[T] := mCurs;
    ModelHandle[T] := mHand;
    FitErr[T] := mHand - MouseVal[T];
    PredictErr[T] := MouseVal[T] - TargetVal[T];
    Inc(T);
  end;
end;

procedure TAnalysisForm.FitParameter(var Param: Real; pMin,pMax: Integer);
var
  Deltap, Range, BestParam, MinErr: double;
  Count: Integer;
begin
  OldRMS := ErrFit;
  Param := pMin;
  Range := (pMax - pMin);
  BestParam := Param;
  RunModel;
  CalcError;
  OldRMS := ErrFit;
  DeltaP := Range/2.0;
  MinErr := 1e6;
  Count := 0;
  repeat
    if OldRMS < ErrFit then DeltaP := -DeltaP/5.0;
    Param := Param + DeltaP;
    RunModel;
    OldRMS := ErrFit;
    CalcError;
    if ErrFit < MinErr then
    begin
      MinErr := ErrFit;
      BestParam := Param;
    end;
    Inc(Count);
  until (Abs(DeltaP) <= 0.001) or (Count > 20);
  Param := BestParam;
end;

procedure TAnalysisForm.FitAll;
var
  I: Integer;
begin
  for I := 1 to 6 do ErrorList[I] := 1e6;
    // Fit output gain
  with OutGainUpDown do
  begin
    FitParameter(GainSet, Min, Max);  // editgain.minvalue etc.
    Position := Round(GainSet);
  end;
  // Fit reference
  with RefUpDown do
  begin
    FitParameter(RefSet, Min, Max);
    Position := Round(RefSet);
  end;
  // Fit delay
  with InDelayUpDown do
  begin
    FitParameter(DelaySet, Min, Max);
    Position := Round(DelaySet);
  end;
  // Fit damping
  with DampUpDown do
  begin
    FitParameter(DampSet, Min, Max);
    Position := Round(DampSet);
  end;
end;

procedure TAnalysisForm.SaveParameters;
begin
  SaveDialog1.Filename := ChangeFileExt(FullPath, '.par');
  if SaveDialog1.Execute then
  begin
    FullParPath := SaveDialog1.FileName;
    ParFileName := ExtractFileName(FullParPath);
    ParamFilenameEdit.Text := ParFileName;
    if FileExists(FullParPath) then
        if MessageDlg(ParFilename + ' already exists.  Overwrite?',
          mtConfirmation, [mbYes, mbNo], 0) = mrNo then
          begin
            ParamFilenameEdit.Text := 'Save aborted';
            Exit;
          end
        else ParamFilenameEdit.Text := 'Overwriting';
  end;
  SaveTheParameters;
end;

procedure TAnalysisForm.SaveMultiRunParameters;
begin
  FullParPath := ChangeFileExt(FullPath, '.par');
  SaveTheParameters;
end;

procedure TAnalysisForm.SaveTheParameters;
begin
  AssignFile(MyFile, FullParPath);
  try
    Rewrite(MyFile);
    WriteLn(MyFile,'TrackParameters');
    WriteLn(MyFile, FullParPath);
    WriteLn(MyFile, DiffLevel:8, DistNo:8);
    WriteLn(MyFile, DelaySet:10:4, (GainSet/10):10:4,
      (DampSet/1000):10:6, (RefSet/10):10:4);
    WriteLn(MyFile, ErrFit:10:3, ErrPred:10:3);
    CloseFile(MyFile);
    ParamFilenameEdit.Text := ParFilename;
  except
    on EINOUTError do
      MessageDlg('File I/O error.', mtError, [mbOK],0);
  end;
end;

procedure TAnalysisForm.ReadParameters;
var
  S: String;
begin
  with OpenDialog1 do
  begin
    Filter := 'Parameter files (*.par)|*.par';
    DefaultExt := 'par';
    Filename := '';
  end;
  if OpenDialog1.Execute then
  begin
    FullParPath := OpenDialog1.FileName;
    ParFileName := ExtractFilename(FullPath);
    if FileExists(FullParPath) then
      begin // Open and read the file
        AssignFile(MyFile, FullParPath);
        try
          Reset(MyFile);
          ReadLn(MyFile, S);
          if S <> 'TrackParameters' then
            MessageDlg(ParFilename + ' is not a valid parameter file.',
              mtError, [mbOK], 0)
          else
            begin
              ParamFilenameEdit.Text := ParFilename;
              ReadLn(MyFile, S);
              ReadLn(MyFile, DiffLevel, DistNo);
              ReadLn(MyFile, DelaySet, GainSet,
                DampSet, RefSet);
              ReadLn(MyFile, ErrFit, ErrPred);
              GainSet := GainSet*10.0;
              DampSet := DampSet*1000.0;
              RefSet := RefSet*10.0;
              InDelayUpDown.Position := Round(DelaySet);
              OutGainUpDown.Position := Round(GainSet);
              DampUpDown.Position := Round(DampSet);
              RefUpDown.Position := Round(RefSet);
            end;
            CloseFile(MyFile);
          except
            on EINOUTError do
              MessageDlg('File I/O error.', mtError, [mbOK],0);
          end;
      end
    else MessageDlg('File not found.', mtInformation,
      [mbOk], 0);
  end;
end;

procedure TAnalysisForm.ReadData;
var
  I, J: Integer;
  S: String;
begin
  with OpenDialog1 do
  begin
    Filter := 'Text files (*.txt)|*.txt';
    DefaultExt := 'txt';
  end;
  if OpenDialog1.Execute then
  begin
    FullPath := OpenDialog1.FileName;
    TxtFileName := ExtractFilename(FullPath);
    if FileExists(FullPath) then
      begin // Open and read the file
        AssignFile(MyFile, FullPath);
        try
          Reset(MyFile);
          ReadLn(MyFile, S);
          if S <> 'TrackData' then
            MessageDlg(TxtFilename + ' is not a valid data file.',
              mtError, [mbOK], 0)
          else
            begin
              FileEdit.Text := TxtFilename;
              ReadLn(MyFile, DiffLevel, DistNo);
              J := 0;
              while not EOF(MyFile) do
              begin
                Inc(J);
                ReadLn(MyFile, I, MouseVal[J], TargetVal[J]);
              end;
              if I <> LastData then
              begin
                MessageDlg(
                'Number of data points not equal to' +
                Format('%4d', [LastData]) +
                '. Load aborted.', mtError, [mbOK], 0);
                Exit;
              end;
              DataPresent := True;
              DiffEdit.Text := Format('%3d', [DiffLevel]);
              DistNoEdit.Text := Format('%3d', [DistNo]);
              for I := 1 to LastData do
              begin
                ErrorVal[I] := TargetVal[I] - MouseVal[I];
                ErrorTimesFive[I] := ErrorVal[I] * 5.0;
              end;
             end;
            CloseFile(MyFile);
          except
            on EINOUTError do
              MessageDlg('File I/O error.', mtError, [mbOK], 0);
          end;
      end
    else MessageDlg('File not found.', mtError,
      [mbOk], 0);
  end;
end;

procedure TAnalysisForm.PlotRunGraph(Color: Integer;
                                     var Data: DataArray; YRange: Real;
                                     Bitmap: TBitmap);
var
  I, Xpos, Ypos, Middle: Integer;
begin
  Middle := Bitmap.Height div 2;
  with Bitmap.Canvas do
  begin
    Pen.Color := Color;
    Ypos := Middle - Round(Data[1]*Bitmap.Height/(YRange));
    MoveTo(0, Ypos);
    for I := 1 to LastData do
    begin
      XPos := I*Bitmap.Width div LastData;
      YPos := Middle - Round(Data[I]*Bitmap.Height/(YRange));
      LineTo(XPos, YPos);
    end;
  end;
end;

procedure TAnalysisForm.PlotRun;
var
  MidScale: Integer;
  PlotRange: Real;
begin
  PlotRange := 1.2 * TargetRange;
  //Plot position
  MidScale := RunBitmap.Height div 2;
  with RunBitmap.Canvas do // Draw the zero-liine.
  begin
     FillRect(Rect(0, 0, RunBitmap.Width, RunBitmap.Height));
     Pen.Color := clGray;
     MoveTo(0, MidScale);
     LineTo(RunBitmap.Width, MidScale);
  end;
  PlotRunGraph(clRed, TargetVal, PlotRange, RunBitmap);
  PlotRunGraph(clLime, MouseVal, PlotRange, RunBitmap);
  if TimesFive then
    PlotRunGraph(clBlack, ErrorTimesFive, PlotRange, RunBitmap)
  else PlotRunGraph(clBlack, ErrorVal, PlotRange, RunBitmap);
  //Show the graphs
  RunPaintBox.Canvas.Draw(0, 0, RunBitmap);
  Stage := 3;
end;

procedure TAnalysisForm.PlotModel(PlotRange: Real);
var
  I, Xpos, Mpos, Midscale: Integer;
  Error: Real;
begin
  with ModelBitmap.Canvas do
  begin
    FillRect(Rect(0, 0, ModelBitmap.Width, ModelBitmap.Height));
    Midscale := ModelBitmap.Height div 2;
    //Draw zero-line
    Pen.Color := clGray;
    MoveTo(0, Midscale);
    LineTo(ModelBitmap.Width, Midscale);
    //Plot actual mouse position
    Pen.Color := clLime;
    Mpos := Midscale - Round(MouseVal[1]
      * ModelBitmap.Height/(PlotRange));
    MoveTo(0, Mpos);
    for I := 1 to LastData do
    begin
      XPos := I*ModelBitmap.Width div LastData;
      Mpos := (ModelBitmap.Height div 2) - Round(MouseVal[I]
        * ModelBitmap.Height/PlotRange);
      LineTo(XPos, Mpos);
    end;
    //Plot model mouse position
    Pen.Color := clBlue;
    Mpos := Midscale - Round(ModelHandle[1]
      * ModelBitmap.Height/(PlotRange));
    MoveTo(0, Mpos);
    for I := 1 to LastData do  //Plot model data
    begin
      XPos := I*ModelBitmap.Width div LastData;
      Mpos := ModelBitmap.Height div 2 - Round(ModelHandle[I]
        * ModelBitmap.Height/PlotRange);
      LineTo(XPos, Mpos);
    end;
    // Plot error
    Pen.Color := clBlack;
    Error := MouseVal[1] - ModelHandle[1];
    if TimesFive then Error := Error * 5.0;
    Mpos := Midscale - Round(Error
      * ModelBitmap.Height/(PlotRange));
    MoveTo(0, Mpos);
    for I := 1 to LastData do  //Plot error data
    begin
      XPos := I*ModelBitmap.Width div LastData;
      Error := MouseVal[I] - ModelHandle[I];
      if TimesFive then Error := Error * 5.0;
      Mpos := ModelBitmap.Height div 2 - Round(Error
        * ModelBitmap.Height/PlotRange);
      LineTo(XPos, Mpos);
    end;
  end;
  ModelPaint.Canvas.Draw(0, 0, ModelBitmap);
end;

procedure TAnalysisForm.CloseBtnClick(Sender: TObject);
begin
  Close;
end;

procedure TAnalysisForm.FormCreate(Sender: TObject);
begin
  ChangeScale(Screen.Height, 768);
  RunBitmap := TBitmap.Create;
  with RunBitmap, Canvas do
  begin
    Height := RunPaintBox.Height;
    Width := RunPaintBox.Width;
    Brush.Color := clWhite;
    Pen.Width := 1;
  end;

  ModelBitmap := TBitmap.Create;
  With ModelBitmap.Canvas do
  begin
    Brush.Color := clWhite;
    Pen.Width := 1;
  end;
  with OpenDialog1 do
  begin
    Filter := 'Text files (*.txt)|*.txt';
    DefaultExt := 'txt';
  end;
  with SaveDialog1 do
  begin
    Filter := 'Parameter files (*.par)|*.par';
    DefaultExt := 'par';
  end;
  dt := 1.0/60.0;
  TimesFive := False;
end;

procedure TAnalysisForm.FormDestroy(Sender: TObject);
begin
  RunBitmap.Free;
  ModelBitmap.Free;
end;

procedure TAnalysisForm.FormPaint(Sender: TObject);
begin
  ModelBitmap.Height := ModelPaint.ClientHeight;
  ModelBitmap.Width := ModelPaint.ClientWidth;
end;

procedure TAnalysisForm.InDelayUpDownClick(Sender: TObject;
  Button: TUDBtnType);
begin
  DelaySet := InDelayUpDown.Position;
end;

procedure TAnalysisForm.OutGainUpDownClick(Sender: TObject;
  Button: TUDBtnType);
begin
  GainSet := OutGainUpDown.Position;
end;

procedure TAnalysisForm.DampUpDownClick(Sender: TObject;
  Button: TUDBtnType);
begin
  DampSet := DampUpDown.Position;
end;

procedure TAnalysisForm.RefUpDownClick(Sender: TObject;
  Button: TUDBtnType);
begin
  RefSet := RefUpDown.Position;
end;

procedure TAnalysisForm.AutoFitBtnClick(Sender: TObject);
var
  I: Integer;
begin
  for I := 1 to 5 do FitAll;
end;

procedure TAnalysisForm.OpenBtnClick(Sender: TObject);
begin
  ReadData;
end;

procedure TAnalysisForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  Stage := 1;
  MainForm.Status.Color := clYellow;
  MainForm.Status.Text := 'Inactive';
end;

procedure TAnalysisForm.SaveParamBtnClick(Sender: TObject);
begin
  SaveParameters;
end;

procedure TAnalysisForm.RetrieveParmsBtnClick(Sender: TObject);
begin
  ReadParameters;
end;

procedure TAnalysisForm.Rescale1Click(Sender: TObject);
begin
  if TimesFive then
    begin
      TimesFive := False;
      Label13.Caption := 'Error';
      Label3.Caption := 'Error';
    end
  else
    begin
      TimesFive := True;
      Label13.Caption := 'Error X 5';
      Label3.Caption := 'Error X 5';
    end;
end;

end.
