unit DisturbFR;

interface

uses SysUtils, Classes, Dialogs;

const
  MaxData = 3599;
  MaxLongTable = MaxData + 3600;

type
  TDisturb = class
    private
      Index: Integer;
      D1, D2, D3, dSlow, dRange: Real;
      DTable, TTable: array[0..MaxData] of Real;
      LongTable: array[0..MaxLongTable] of Real;
      FileS: TFilestream;
    public
      DisTable: array[1..5] of array[1..10]
        of array[1..MaxData+1] of Real;
      procedure NewTable; Overload;
      procedure NewTable(Slow, Range: Real); Overload;
      function NextValue: Real;
      function Range: Real;
      function Slow: Real;
      procedure MakeDisturbanceTable(Range: Real);
      procedure ReadDisTable;
      procedure SaveDisTable;
    end;

implementation

procedure TDisturb.NewTable;
var
  Max, Min: Real;
  I, J: Integer;
begin
  Index := 0;
  repeat
    // make long disturbance table
    D1 := 0.0; D2 := 0.0; D3 := 0.0;
    for I := 0 to MaxLongTable do
    begin
      D1 := D1 + dSlow*(20000*(Random - 0.5));
      D2 := D2 + dSlow*(D1 - D2);
      D3 := D3 + dSlow*(D2 - D3);
      LongTable[I] := D3;
    end;
            // find first place where data[i] = data[i+3599] within 50
    I := 0;
    if (LongTable[I] >= LongTable[I+MaxData]) then
      while not (LongTable[i] <= LongTable[I+MaxData]) or
         (I+MaxData < MaxLongTable)  do Inc(I)
    else
      while not ((LongTable[I] >= LongTable[I+MaxData]) or
         (I+MaxData > MaxLongTable)) do Inc(I);
  until (I+MaxData < MaxLongTable);

  // transfer to output table
  for J := I to I+MaxData do TTable[J-I] := LongTable[J];
  //Scale output table to make max - min value = 600;
  Max := -1e6;
  Min := 1e6;
  for J := 0 to MaxData do
  begin
    if TTable[J] > Max then Max := TTable[J];
    if TTable[J] < Min then Min := TTable[J];
  end;
  for J := 0 to MaxData do TTable[J] := TTable[J] - (Max + Min)/2;
  // center within max - min range
  for I := 0 to MaxData do TTable[I] := TTable[I]/(Max - Min)*dRange;
  // find first zero crossing;
  I := 0;
  While TTable[I]*TTable[I+1] > 0 do Inc(I);
  // transfer to dtable with zero at start
  j := 0;
  while J <= MaxData do
  begin
    DTable[J] := TTable[I];
    I := (I + 1) mod (MaxData + 1);
    Inc(J);
  end;
end;

procedure TDisturb.NewTable(Slow, Range: Real);
begin
  dSlow := Slow;
  dRange := Range;
  NewTable;
end;

function TDisturb.NextValue: Real;
begin
  NextValue := DTable[Index];
  if Index = MaxData then
    begin
      Index := 0;
      NewTable;
    end else Inc(Index);
end;

function TDisturb.Range: Real;
begin
  Range := dRange;
end;

function TDisturb.Slow: Real;
begin
  Slow := dSlow;
end;

procedure TDisturb.MakeDisturbanceTable(Range: Real);
var
  I, J, K: Integer;
  Slow: Real;
begin
  for I := 1 to 5 do
  begin
    Case I of
      1: Slow := 64;
      2: Slow := 32;
      3: Slow := 16;
      4: Slow := 8;
      5: Slow := 4;
    end;
    Slow := 1.0/Slow;
    for J := 1 to 10 do
    begin
      NewTable(Slow, Range);
      for K := 1 to MaxData+1 do
        DisTable[I, J, K] := NextValue;
    end;
  end;
end;

procedure TDisturb.SaveDisTable;
var
  I, J, K: Integer;
  Buff: Real;
  MyFile: TFilename;
begin
  MyFile := 'Distable';
  FileS := TFileStream.Create(MyFile, fmCreate);
  try
    for I := 1 to 5 do
      for J := 1 to 10 do
        for K := 1 to MaxData+1 do
        begin
          Buff := DisTable[I, J, K];
          FileS.WriteBuffer(Buff, SizeOf(Buff));
        end;
  finally
    FileS.Free;
  end;
end;

procedure TDisturb.ReadDisTable;
var
  I, J, K: Integer;
  Buff: Real;
  MyFile: TFilename;
begin
  MyFile := 'Distable';
  FileS := TfileStream.Create(MyFile, fmOpenRead, fmShareDenyWrite);
  try
    for I := 1 to 5 do
      for J := 1 to 10 do
        for K := 1 to MaxData+1 do
        begin
          FileS.ReadBuffer(Buff, SizeOf(Buff));
          DisTable[I, J, K] := Buff;
        end;
  finally
    FileS.Free;
  end;
end;



end.
