unit Main;

//Program: TrackAnalyze.exe
//Version: 9/13/2008
//Authors: William T. Powers and Bruce B. Abbott
//Copyright: 2008

interface

uses
  Windows, Messages, SysUtils, StrUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, ComCtrls, DisturbFR, TargetUnit;

const
  LastData = 3600;

type
  DataArray = array[1..LastData] of Real;
  DistTableArray = array[1..5] of array[1..10] of DataArray;

  TMainForm = class(TForm)
    MainPanel: TPanel;
    SaveDialog1: TSaveDialog;
    ControlPanel: TGroupBox;
    DataOptionsBox: TGroupBox;
    AnalyzeBtn: TButton;
    CollectDataBtn: TButton;
    ProgramBox: TGroupBox;
    InstructionBtn: TButton;
    AboutBtn: TButton;
    ExitBtn: TButton;
    TargetBox: TGroupBox;
    TargetPaint: TPaintBox;
    RecordBox: TGroupBox;
    Label1: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label8: TLabel;
    StartBtn: TButton;
    AbortBtn: TButton;
    Status: TEdit;
    ETCounter: TEdit;
    SaveDataEdit: TEdit;
    CurrentRunEdit: TEdit;
    RunSettingsBox: TGroupBox;
    Label5: TLabel;
    Label6: TLabel;
    Label7: TLabel;
    Label2: TLabel;
    Label9: TLabel;
    DiffEdit: TEdit;
    DisturbEdit: TEdit;
    NRunsEdit: TEdit;
    DiffUpDown: TUpDown;
    DisturbUpDown: TUpDown;
    NRunsUpDown: TUpDown;
    IRIEdit: TEdit;
    IRIUpDown: TUpDown;
    procedure TargetPaintMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure StartBtnClick(Sender: TObject);
    procedure AbortBtnClick(Sender: TObject);
    procedure AnalyzeBtnClick(Sender: TObject);
    procedure CollectDataBtnClick(Sender: TObject);
    procedure AboutBtnClick(Sender: TObject);
    procedure InstructionBtnClick(Sender: TObject);
    procedure ExitBtnClick(Sender: TObject);
    procedure DiffUpDownClick(Sender: TObject; Button: TUDBtnType);
    procedure DisturbUpDownClick(Sender: TObject; Button: TUDBtnType);
    procedure NRunsUpDownClick(Sender: TObject; Button: TUDBtnType);
    procedure IRIUpDownClick(Sender: TObject; Button: TUDBtnType);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure TargetPaintPaint(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    procedure Initialize;
    procedure FitComponents;
    procedure OperateTarget;
    procedure RecordData;
    procedure GetFilename;
    procedure IndexFileName;
    procedure SaveData;
    procedure IdleLoop(Sender: TObject; var Done: boolean);
  end;

var
  MainForm: TMainForm;
  Offscreen: TBitmap;
  D1: TDisturb;
  DisTable: DistTableArray;
  Target: TTarget;
  MouseVal, TargetVal, ModelMouseVal, ErrorVal: DataArray;
  DataPresent, MultiRuns: Boolean;
  Stage: Integer;
  Stem: String;

implementation

uses Analysis, About, Instructions;

{$R *.dfm}

var
  T1, T2, Freq, Elapsed, StartDelay: Int64;
  DiffLevel, DistNo, IndexNo, NRuns, CurrentRun, TimeCount: Integer;
  Running: Boolean;

// ------------------------- Initialization -----------------------

procedure ZeroDataArrays;
var
  I: Integer;
begin
  for I := 1 to LastData do
  begin
    MouseVal[I] := 0;
    TargetVal[I] := 0;
    ErrorVal[I] := 0;
    ModelMouseVal[I] := 0;
  end;
end;

procedure TMainForm.Initialize;  // See also Stage 0 of procedure IdleLoop
begin
  Randomize;
  TargetRange := 1000;
  Stage := 0;
  DiffLevel := 3;
  DistNo := 1;
  IndexNo := 1;
  Running := False;
  MultiRuns := False;
  StartDelay := 240;
  if FileExists('Distable') then D1.ReadDisTable
  else
    begin
      D1.MakeDisturbanceTable(TargetRange);
      D1.SaveDisTable;
    end;
  TargetRange := 1200;
  DataPresent := False;
  with SaveDialog1 do
  begin
    Filter := 'Text files (*.txt)|*.txt';
    DefaultExt := 'txt';
    Title := 'Save Data File As'
  end;
  ZeroDataArrays;
end;

// -------------- Datafile Creation and Data Saving ---------------

procedure TMainForm.GetFilename;
begin
  if SaveDialog1.Execute then
  begin
    FullPath := SaveDialog1.FileName;
    TxtFileName := ExtractFileName(FullPath);
    if FileExists(FullPath) then
      if MessageDlg('File exists.  Overwrite?',
        mtConfirmation, [mbYes, mbNo], 0) = mrNo then
        begin
          TxtFilename := '';
          Exit;
        end;
    //IndexFilename;
    SaveDataEdit.Text := TxtFileName;
  end
end;

procedure TMainForm.IndexFilename;
var
  ASCII: Byte;
  I: Integer;
  Stem, NewExt: String;
begin
  Stem := ChangeFileExt(ExtractFilename(FullPath), '');
  I := Pos('_', Stem);                    //I > 0 if Stem contains '_' char
  if I > 0 then Stem := LeftStr(Stem,  I-1); //Strip index if present
  ASCII := 96 + CurrentRun;
  NewExt := '_' + Chr(ASCII) + '.txt';     //Add new index and file ext
  FullPath := ExtractFilePath(FullPath)+ Stem + NewExt;
  SaveDialog1.FileName := FullPath;
  TxtFileName := ExtractFileName(FullPath);
end;

procedure TMainForm.SaveData;
var
  I: Integer;
begin
  AssignFile(MyFile, FullPath);
  try
    Rewrite(MyFile);
    WriteLn(MyFile,'TrackData');
    WriteLn(MyFile, DiffLevel:8, DistNo:8);
    for I := 1 to LastData do
      WriteLn(MyFile, I:6, MouseVal[I]:10:2, TargetVal[I]:10:2);
    CloseFile(MyFile);
    SaveDataEdit.Text := TxtFilename;
  except
    on EINOUTError do
      MessageDlg('File I/O error.', mtError, [mbOK],0);
  end;
end;

// ------------- Operate Target and Cursor, Record Run Data --------------

procedure TMainForm.OperateTarget;
begin
  OffScreen.Canvas.FillRect(Rect(0, 0, Offscreen.Width, Offscreen.Height));
  TargetPos := D1.DisTable[DiffLevel, DistNo, IndexNo]; // D1.NextValue;
  inc(IndexNo);
  if IndexNo > LastData then IndexNo := 1;
  Target.DrawTarget(OffScreen.Height div 2 -
    Round(TargetPos*OffScreen.Height/TargetRange), OffScreen);
  Target.DrawCursor(OffScreen.Height div 2 -
    Round(CursorPos*OffScreen.Height/TargetRange), Offscreen);
  TargetPaint.Canvas.Draw(0, 0, OffScreen);
end;

procedure TMainForm.TargetPaintMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
// Set cursor Y via mouse Y if mouse is in target display
begin
  CursorPos := (OffScreen.Height div 2 - Y)*TargetRange/(OffScreen.Height);
  if CursorPos > 500 then CursorPos := 500
    else if CursorPos < -500 then CursorPos := -500;
end;

procedure TMainForm.RecordData;
var
  ET: Integer;
begin
  Inc(Elapsed);
  ET := Round(Elapsed/60.0);
  ETcounter.Text := Format('%3d',[ET]);
  if Elapsed = 0 then
    begin
      Status.Color := clLime;
      Status.Text := 'Recording Data';
    end
  else if Elapsed > 0 then // Elapsed starts < 0 to allow time to grab cursor.
    begin
      TargetVal[Elapsed] := TargetPos;
      MouseVal[Elapsed] := CursorPos;
      ErrorVal[Elapsed] := (TargetPos-CursorPos);
    end;
  if Elapsed >= LastData then //End run at 60 seconds
  begin
    Elapsed := -StartDelay;
    Running := False;
    DataPresent := True;
    SaveData;

    if MultiRuns then // Auto-fit and save parameters of each run
      begin
        AnalysisForm.FitAll;
        AnalysisForm.SaveMultiRunParameters;
      end;

    if (MultiRuns) and (CurrentRun < NRuns) then Stage := 4
    else
      begin
        Stage := 1;
        Status.Text := 'Idle';
        Status.Color := clYellow;
      end;
  end;
end;

// ------- Main Program Loop, Executed 60 times per second ----------

procedure TMainForm.IdleLoop(Sender: TObject; var Done: boolean);
begin
  Done := false;
  Case Stage of
  0: begin // Initialize
       MainPanel.Left := MainForm.Width div 2 - MainPanel.Width div 2;
       MainPanel.Top := MainForm.Top; MainPanel.Height := MainForm.ClientHeight;
       TargetBox.Height := Round (0.95*MainPanel.Height);
       ControlPanel.Height := TargetBox.Height;
       OffScreen.Width := TargetPaint.ClientWidth;
       OffScreen.Height := TargetPaint.ClientHeight;
       Target.Initialize(OffScreen.Width div 2, OffScreen.Height div 2, 1,
         clRed, clLime);
       with Offscreen.Canvas do
       begin
         Brush.Color := clBlack;
         FillRect(Rect(0, 0, Offscreen.Width, Offscreen.Height));
       end;
       TargetPaint.Canvas.Draw(0, 0, OffScreen);
       Status.Text := 'Inactive';
       Status.Color := clYellow;
       SaveDataEdit.Text := 'Default.txt';
       NRuns := NRunsUpDown.Position;
       CurrentRun := 1;
       CurrentRunEdit.Text := Format('%3d', [CurrentRun]);
       Stage := 1;
     end;
  1: begin // Idle state
       with Offscreen.Canvas do
       begin
         FillRect(Rect(0, 0, Offscreen.Width, Offscreen.Height));
         TargetPaint.Canvas.Draw(0, 0, OffScreen);
       end;
     end;
  2: begin // Practice or data-collection state
       OperateTarget;
       if Running then RecordData;
     end;
  3: begin // Data analysis state
       if DataPresent then
         with AnalysisForm do
         begin
           PlotRun;
           InitAnModel;
           RunModel;
           CalcError;
           PlotModel(1.2*1000);
         end;
     end;
  4: begin // Multi-run; Initialize next Run
       Inc(CurrentRun);
       IndexFilename;
       SaveDataEdit.Text := ExtractFilename(FullPath);
       CurrentRunEdit.Text := Format('%3d', [CurrentRun]);
       TimeCount := IRIUpDown.Position * 60;
       with Offscreen.Canvas do
       begin
         FillRect(Rect(0, 0, Offscreen.Width, Offscreen.Height));
         TargetPaint.Canvas.Draw(0, 0, OffScreen);
       end;
       Running := True;
       Stage := 5;
     end;
  5: begin // Time inter-run interval
       Dec(TimeCount);
       if (TimeCount Mod 60 = 0) then
         ETCounter.Text := Format('%3d', [TimeCount div 60]);
       if TimeCount = 0 then
         begin
           Elapsed := -StartDelay;
           Status.Text := 'Start Tracking';
           IndexNo := LastData - StartDelay; // Reset to last 4 sec of
                   // disturbance waveform. Disturbance will wrap to beginning
                   // at start of recording.
           Stage := 2; // Start next run
         end;
     end;
  end;
  repeat
    QueryPerformanceCounter(T2);
  until (T2 - T1) >= Freq/60;
  T1 := T2;
  Application.ProcessMessages;
end;

// -------------------- Data Recorder Controls ---------------------

procedure TMainForm.StartBtnClick(Sender: TObject);
begin
  if Running then Exit; // Don't restart in mid-run.
  if Status.Text = 'Inactive' then
  begin
    MessageDlg(
      'First press "Collect Data" button to enter data collection mode.',
      mtInformation, [mbOK], 0);
    Exit;
  end;
  Elapsed := -StartDelay;
  IndexNo := LastData - StartDelay; // Reset to last 4 sec of disturbance
        // waveform. Disturbance will wrap to beginning at start of recording.
  Status.Text := 'Start Tracking';
  Running := True;
  Stage := 2;
end;

procedure TMainForm.AbortBtnClick(Sender: TObject);
begin
  if Status.Text = 'Inactive' then Exit;
  DataPresent := False;
  ZeroDataArrays;
  Status.Color := clYellow;
  if NRuns = 1 then // Abort run and return to practice state
    begin
      ETCounter.Text := ' ';
      Status.Text := 'Pre-run Practice';
      Running := False;
      Stage := 2;
    end
  else  // Multiple runs in progress
    if MessageDlg('Cancel remaining multiple runs?', mtConfirmation,
      [mbYes, mbNo], 0) = mrYes then
      begin // Return to idle state to force new datafile selection
        CurrentRun := 1;
        CurrentRunEdit.Text := Format('%3d', [CurrentRun]);
        Status.Text := 'Inactive';
        Running := False;
        Stage := 1;
      end
    else // Continue run series; start recording same run again.
      begin
        Elapsed := -StartDelay;
        IndexNo := LastData - StartDelay; // Reset to last 4 sec of disturbance
        // waveform. Disturbance will wrap to beginning at start of recording.
        Status.Text := 'Start Tracking';
        Running := True;
        Stage := 2;
      end;
end;

// ------------------------ Data Options Controls --------------------

procedure TMainForm.CollectDataBtnClick(Sender: TObject);
begin
  if (NOT Running) then
  begin
    GetFilename;
    if TxtFilename <> '' then
      begin
        Status.Text := 'Pre-run Practice';
        Stage := 2;
      end
    else
      MessageDlg('No data filename specified; Data collection cancelled.',
        mtError, [mbOK], 0);
  end;
end;

procedure TMainForm.AnalyzeBtnClick(Sender: TObject);
var
  I: Integer;
begin
  if Running then Exit;
  Stage := 3;
  if DataPresent then
    with AnalysisForm do
    begin
      DiffEdit.Text := Format('%3d', [DiffLevel]);
      DistNoEdit.Text := Format('%3d', [DistNo]);
    end;
  with AnalysisForm do
  begin
    if TxtFileName = '' then FileEdit.Text := 'Default.txt'
      else FileEdit.Text := TxtFileName;
    for I := 1 to LastData do // Compute magnified error array
      ErrorTimesFive[I] := ErrorVal[I] * 5.0;
    ShowModal;
  end;
end;

// -------------------- Program Options Controls ------------------

procedure TMainForm.AboutBtnClick(Sender: TObject);
begin
  AboutBox.ShowModal;
end;

procedure TMainForm.InstructionBtnClick(Sender: TObject);
begin
  if NOT Running then InstructForm.ShowModal;
end;

procedure TMainForm.ExitBtnClick(Sender: TObject);
begin
  Application.Terminate;
end;

// -------------------- Run Parameters Box Controls ------------------

procedure TMainForm.DiffUpDownClick(Sender: TObject; Button: TUDBtnType);
begin
  DiffLevel := DiffUpDown.Position;
end;

procedure TMainForm.DisturbUpDownClick(Sender: TObject;
  Button: TUDBtnType);
begin
  DistNo := DisturbUpDown.Position;
end;

procedure TMainForm.NRunsUpDownClick(Sender: TObject; Button: TUDBtnType);
begin
  NRuns := NRunsUpDown.Position;
  if NRuns > 1 then MultiRuns := True else MultiRuns := False;
end;

procedure TMainForm.IRIUpDownClick(Sender: TObject; Button: TUDBtnType);
begin
  If NOT Running then
  begin
    TimeCount := IRIUpDown.Position;
    CurrentRun := 0;
  end;
end;

// ----------------  MainForm Create, Destroy, Resize, Paint --------------

procedure TMainForm.FormCreate(Sender: TObject);
var
  ScreenRatio: Real;
begin
  OffScreen := TBitmap.Create;
  //Set to nonzero values to prevent divide by zero in mouse routine.
  //Will be changed to correct values in Stage 0 of iterate procedure.
  OffScreen.Height := 200;
  OffScreen.Width := 300;
  
  ScreenRatio := Screen.Width/Screen.Height;
  if ScreenRatio > 0.75 then ChangeScale(Screen.Width, 1024)
    else ChangeScale(Screen.Height, 768);
  WindowState := wsMaximized;
  D1 := TDisturb.Create;
  Target := TTarget.Create;
  Initialize;
//Get timer info and establish idle loop;
  QueryPerformanceFrequency(Freq);
  QueryPerformanceCounter(T1);
  Application.OnIdle := IdleLoop;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  Target.Free;
  D1.Free;
  Offscreen.Free;
end;

procedure TMainForm.FitComponents;
begin
  OffScreen.Height := TargetPaint.Height;
  OffScreen.Width := TargetPaint.Width;
end;

procedure TMainForm.TargetPaintPaint(Sender: TObject);
begin
  TargetPaint.Canvas.Draw(0, 0, OffScreen);

end;

end.
